import numpy as np
import matplotlib.pyplot as plt

# Golden ratio constant
phi = (1 + np.sqrt(5)) / 2

# First 50 primes for entropy injection
PRIMES = [
    2, 3, 5, 7, 11, 13, 17, 19, 23, 29,
    31, 37, 41, 43, 47, 53, 59, 61, 67, 71,
    73, 79, 83, 89, 97, 101, 103, 107, 109, 113,
    127, 131, 137, 139, 149, 151, 157, 163, 167, 173,
    179, 181, 191, 193, 197, 199, 211, 223, 227, 229
]

def fib_real(n):
    """Generalized Fibonacci real extension using Binet's formula."""
    from math import cos, pi, sqrt
    phi_inv = 1 / phi
    return (phi**n / sqrt(5)) - (phi_inv**n) * cos(pi * n)

def D(n, beta, r=1.0, k=1.0, Omega=1.0, base=2):
    """Dimensional DNA operator for scale n+beta."""
    Fn_beta = fib_real(n + beta)
    idx = int(np.floor(n + beta) + len(PRIMES)) % len(PRIMES)
    Pn_beta = PRIMES[idx]
    dyadic = base ** (n + beta)
    val = phi * Fn_beta * dyadic * Pn_beta * Omega
    val = np.maximum(val, 1e-15)  # Avoid sqrt of negative
    return np.sqrt(val) * (r ** k)

# Define recursive scaling of time, length, charge
def time_scale(n):
    return phi ** n  # s = φ^n

def length_scale(n, Omega=1.0):
    # m = sqrt(Ω * φ^{7n})
    return np.sqrt(Omega * phi ** (7 * n))

def charge_scale(n):
    # C = s^3 = φ^{3n}
    return phi ** (3 * n)

def force_scale(n, Omega=1.0):
    # F = E/m ~ sqrt(Ω) * φ^{1.5 n}
    return np.sqrt(Omega) * phi ** (1.5 * n)

def speed_of_light(n, Omega=1.0):
    # c = m / s = sqrt(Ω) * φ^{2.5 n}
    return np.sqrt(Omega) * phi ** (2.5 * n)

# Example parameters
Omega = 1.0  # normalized field tension
k = 1.0      # spatial exponent power
r = 1.0      # length unit

# Example recursion depth range (cosmic unfolding)
n_values = np.linspace(-1, 2, 200)  # sample from smaller to larger scale

# Compute emergent c over recursion depth n
c_values = speed_of_light(n_values, Omega)

plt.figure(figsize=(10,6))
plt.plot(n_values, c_values, label=r'Emergent $c = \sqrt{\Omega} \cdot \varphi^{2.5 n}$')
plt.axhline(y=299792.458, color='red', linestyle='--', label='Physical $c$ (km/s)')
plt.xlabel('Recursion Depth $n$')
plt.ylabel('Speed of Light (arbitrary units)')
plt.title('Emergent Speed of Light from Recursive Golden Ratio Scaling')
plt.legend()
plt.grid(True)
plt.show()

print(f"Speed of light at n=0: c = {speed_of_light(0, Omega)}")
print(f"Speed of light at n=1: c = {speed_of_light(1, Omega)}")
print(f"Speed of light at n=2: c = {speed_of_light(2, Omega)}")
